using System;
using gov.va.med.vbecs.Common;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// TestValidationModel
    /// </summary>
    public class TestValidationModel : IEquatable<TestValidationModel>
    {
        /// <summary>
        /// ValidationStatus (Unkown, Valid, Warning, Error)
        /// </summary>
        public TestValidationStatus TestValidationStatus { get; set; }
        /// <summary>
        /// List of override messages
        /// </summary>
        public IList<ExceptionForDisplayModel> ExceptionForDisplayList { get; set; }
        /// <summary>
        /// List of messages to prompt the user with when selecting this test
        /// </summary>
        public IList<StrRes.StringResourceItem> UserWarningPromptList { get; set; }
        /// <summary>
        /// Quarantine this test when accepting
        /// </summary>
        public bool Quarantine { get; set; }
        /// <summary>
        /// ErrorMessage used for tooltip
        /// </summary>
        public string ErrorMessage { get; set; }

        /// <summary>
        /// TestValidationModel
        /// </summary>
        public TestValidationModel()
        {
            ExceptionForDisplayList = new List<ExceptionForDisplayModel>();
            UserWarningPromptList = new List<StrRes.StringResourceItem>();
            Quarantine = false;
        }

        /// <summary>
        /// TestValidationModel constructor
        /// </summary>
        /// <param name="testValidationStatus"></param>
        /// <param name="exceptionForDisplayList"></param>
        /// <param name="errorMessage"></param>
        public TestValidationModel(TestValidationStatus testValidationStatus, IList<ExceptionForDisplayModel> exceptionForDisplayList, string errorMessage)
            : this()
        {
            TestValidationStatus = testValidationStatus;
            ExceptionForDisplayList = exceptionForDisplayList;
            ErrorMessage = errorMessage;
        }

        /// <summary>
        /// TestValidationModel constructor
        /// </summary>
        /// <param name="testValidationStatus"></param>
        /// <param name="exceptionForDisplayList"></param>
        /// <param name="errorMessage"></param>
        /// <param name="quarantine"></param>
        /// <param name="userWarningPromptList"></param>
        public TestValidationModel(
            TestValidationStatus testValidationStatus,
            IList<ExceptionForDisplayModel> exceptionForDisplayList,
            string errorMessage,
            bool quarantine,
            IList<StrRes.StringResourceItem> userWarningPromptList)
            : this(testValidationStatus, exceptionForDisplayList, errorMessage)
        {
            Quarantine = quarantine;
            UserWarningPromptList = userWarningPromptList;
        }

        /// <summary>
        /// Clone TestValidationModel
        /// </summary>
        /// <returns></returns>
        public TestValidationModel Clone()
        {
            IList<ExceptionForDisplayModel> cloneExceptionTypeList = null;
            IList<StrRes.StringResourceItem> cloneUserWarningPromptList = null;

            if (ExceptionForDisplayList != null)
            {
                cloneExceptionTypeList = ExceptionForDisplayList.Select(x => x.Clone()).ToList();
            }

            if (UserWarningPromptList != null)
            {
                cloneUserWarningPromptList = UserWarningPromptList.ToList();
            }

            return new TestValidationModel(TestValidationStatus,
                                           cloneExceptionTypeList,
                                           ErrorMessage,
                                           Quarantine,
                                           cloneUserWarningPromptList);
        }

        /// <summary>
        /// Equals
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(TestValidationModel other)
        {
            if (ReferenceEquals(null, other)) return false;
            if (ReferenceEquals(this, other)) return true;

            var equals = TestValidationStatus == other.TestValidationStatus
                && Quarantine.Equals(other.Quarantine)
                && string.Equals(ErrorMessage, other.ErrorMessage);

            // Compare the ExceptionTypeList
            if (equals)
            {
                if (ExceptionForDisplayList == null &&
                    other.ExceptionForDisplayList != null)
                {
                    equals = false;
                }
                else if (ExceptionForDisplayList != null &&
                         other.ExceptionForDisplayList == null)
                {
                    equals = false;
                }
                else if (ExceptionForDisplayList != null &&
                         other.ExceptionForDisplayList != null)
                {
                    equals = ExceptionForDisplayList.Count == other.ExceptionForDisplayList.Count;

                    if (equals)
                    {
                        equals = ExceptionForDisplayList.SequenceEqual(other.ExceptionForDisplayList);
                    }
                }
            }

            // Compare the UserWarningPromptList
            if (equals)
            {
                if (UserWarningPromptList == null &&
                    other.UserWarningPromptList != null)
                {
                    equals = false;
                }
                else if (UserWarningPromptList != null &&
                         other.UserWarningPromptList == null)
                {
                    equals = false;
                }
                else if (UserWarningPromptList != null &&
                         other.UserWarningPromptList != null)
                {
                    equals = UserWarningPromptList.Count == other.UserWarningPromptList.Count;

                    if (equals)
                    {
                        equals = !UserWarningPromptList.Any(x => other.UserWarningPromptList.Any(o => x.ResString != o.ResString));
                    }
                }
            }
            return equals;
        }

        /// <summary>
        /// Equals
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            if (ReferenceEquals(null, obj)) return false;
            if (ReferenceEquals(this, obj)) return true;
            if (obj.GetType() != this.GetType()) return false;
            return Equals((TestValidationModel)obj);
        }

        /// <summary>
        /// GetHashCode
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            unchecked
            {
                int hashCode = (int)TestValidationStatus;
                hashCode = (hashCode * 397) ^ (ExceptionForDisplayList != null ? ExceptionForDisplayList.GetHashCode() : 0);
                hashCode = (hashCode * 397) ^ (UserWarningPromptList != null ? UserWarningPromptList.GetHashCode() : 0);
                hashCode = (hashCode * 397) ^ Quarantine.GetHashCode();
                hashCode = (hashCode * 397) ^ (ErrorMessage != null ? ErrorMessage.GetHashCode() : 0);
                //hashCode = (hashCode * 397) ^ (OverrideMessageDictionary != null ? OverrideMessageDictionary.GetHashCode() : 0);
                return hashCode;
            }
        }
    }
}



